#include "StdV3D.h"
#include "Basics/Vector.h"
#include "Basics/MatrixTransf.h"

#include <limits>

#include "IsoSurface.h"
#include "MarchCubesByLayers.h"

namespace V3D {


MarchCubesByLayers::MarchCubesByLayers(const Vector3D& vcCenterGrid, const Vector3D& vcSize, 
									   int32 nDivideX, int32 nDivideY, int32 nDivideZ, 
									   int32 nRefineLevel /* = 0 */)
	: parent( vcCenterGrid, vcSize, nDivideX, nDivideY, nDivideZ),
	  m_nRefineIterCount( nRefineLevel),
	  m_bCleanupTriangles( false )
{}


MarchCubesByLayers::~MarchCubesByLayers()
{}


void MarchCubesByLayers::InitGrid( const Vector3D& vcCenterGrid, const Vector3D& vcSize, int32 nDivideX, int32 nDivideY, int32 nDivideZ)
{
	SetGridInfo( vcCenterGrid, vcSize, nDivideX, nDivideY, nDivideZ);
}

void MarchCubesByLayers::SetRefineLevel( int32 nLevel )
{
	m_nRefineIterCount = nLevel;
}

void MarchCubesByLayers::SetCleanupTriangles( bool bCleanup /* = true */)
{
	m_bCleanupTriangles = bCleanup;
}


void MarchCubesByLayers::UpdateBufferSizes(int32 nDivideX, int32 nDivideY)
{
	if( double(nDivideX+1) * double(nDivideY+1) > double( MAX_INT32 ) )
	{
		nDivideX = MAX_INT16 - 1;
		nDivideY = MAX_INT16 - 1;
	}
	
	if( m_aScansPotentialCubesNxt.GetSize() < nDivideY)
	{
		// Initialise le tableau des lignes de cubes a visiter
		m_aScansPotentialCubesNxt.SetSize(nDivideY);
		m_aScansPotentialCubesCur.SetSize(nDivideY);
	}

	if( int32(m_abTesselatedCubeCurLayer.size()) < nDivideX*nDivideY)
	{
		int32 nOldSize = m_abTesselatedCubeCurLayer.size();
		int32 nNewSize = nDivideX*nDivideY;
		m_abTesselatedCubePrvLayer.resize(nNewSize);
		m_abTesselatedCubeCurLayer.resize(nNewSize);
		m_abDummyLayer.resize(nNewSize);

		// Les tableaux de booleens doivent etre vides
		std::fill( m_abTesselatedCubePrvLayer.begin() + nOldSize, m_abTesselatedCubePrvLayer.end(), false);
		std::fill( m_abTesselatedCubeCurLayer.begin() + nOldSize, m_abTesselatedCubeCurLayer.end(), false);
	}

	int32 nPotentialPointsCount = (nDivideX + 1) * (nDivideY + 1);
	if( int32(m_afPotentialsCurLayer.size()) < nPotentialPointsCount )
	{
		// Definition de la taille des tableaux de densites pour la couche courante de cubes
		m_afPotentialsCurLayer.resize(nPotentialPointsCount);
		m_afPotentialsPrvLayer.resize(nPotentialPointsCount);
	}

	// Definition de la taille des tableaux de vertices sur les aretes X des cubes de la couche courante
	int32 nPotentialPointsXAxisCount = nDivideX * (nDivideY + 1);
	if( int32(m_aiVtxIndexXCurLayer.size()) < nPotentialPointsXAxisCount )
	{
		m_aiVtxIndexXCurLayer.resize(nPotentialPointsXAxisCount);   // Z sup de la couche
		m_aiVtxIndexXPrvLayer.resize(nPotentialPointsXAxisCount);   // Z inf de la couche
	}
	// Definition de la taille des tableaux de vertices sur les aretes Y des cubes de la couche courante
	int32 nPotentialPointsYAxisCount = nDivideY * (nDivideX + 1);
	if( int32(m_aiVtxIndexYCurLayer.size()) < nPotentialPointsYAxisCount )
	{
		m_aiVtxIndexYCurLayer.resize(nPotentialPointsYAxisCount);   // Z sup de la couche
		m_aiVtxIndexYPrvLayer.resize(nPotentialPointsYAxisCount);   // Z inf de la couche
	}

	// Definition de la taille des tableaux de vertices sur les aretes Z des cubes de la couche courante
	int32 nPotentialPointsZAxisCount = (nDivideX + 1) * (nDivideY + 1);
	if( int32(m_aiVtxIndexZLayer.size()) < nPotentialPointsZAxisCount )
	{
		m_aiVtxIndexZLayer.resize(nPotentialPointsZAxisCount);
	}
}









void MarchCubesByLayers::GenerateMesh( const IsoSurface& iso,
										  Vect3DArray& aVertices, TriangleArray& aTriangles,
										  Vect3fArray* paNorms /*= 0 */, bool bNormalizeNormals /* = false */ )
{
	// On vide d'abord le tableau des triangles et des vertex
//	aVertices.SetSize(0);
//	aTriangles.SetSize(0);

	int32 nLayerStartX, nLayerStartY, nLayerStartZ;
	int32 nLayerCubesX, nLayerCubesY, nLayerCubesZ;

	ComputeUsefulSubGrid( nLayerStartX, nLayerCubesX, 
	                      nLayerStartY, nLayerCubesY, 
	                      nLayerStartZ, nLayerCubesZ, iso);

	// Si il n'y a pas d'epaisseur, rien ne sera genere de toute facon, on quitte tout de suite
	if( nLayerCubesX == 0 || nLayerCubesY ==0 || nLayerCubesZ == 0 )
		return;

	const Vector3D vcSpaceNegCorner = GetGridPoint(nLayerStartX, nLayerStartY, nLayerStartZ);
	
	int32 nFirstGeneratedVertex = aVertices.size();

	// On lance tout le travail de generation de la geometrie (vertices, triangles)
	GenerateGeometryInGrid( aVertices, aTriangles, iso,
							vcSpaceNegCorner, nLayerCubesX, nLayerCubesY, nLayerCubesZ);

	int32 nNewVerticesCount  = aVertices.size();
	int32 nGeneratedVertices = nNewVerticesCount - nFirstGeneratedVertex;

	// Le calcul des normales se fait ici
	if( nGeneratedVertices > 0 && paNorms != 0)
	{
		Vect3fArray& aNorms = *paNorms;
		
		const Vect3DArray::size_type nVertexCount = aVertices.size();
		if(	aNorms.size() < nVertexCount)
			aNorms.resize(nVertexCount);

//		iso.ComputeSurfaceNormalsSubArray(*paNorms, aVertices, nFirstGeneratedVertex, nGeneratedVertices);
//		iso.ComputeSurfaceNormals(*paNorms, aVertices);
		ComputeNormalsFromTopology(*paNorms, aVertices, aTriangles);

		if( bNormalizeNormals)
		{
			// Normalization une a une des normales
			for( int32 i = nFirstGeneratedVertex; i < nNewVerticesCount; ++i)
			{
				Vector3f& norm = aNorms[i];
				norm.Normalize();
			}
		}
	}
}


void MarchCubesByLayers::GenerateGeometryInGrid( Vect3DArray& aVertices, TriangleArray& aTriangles,
												 const IsoSurface& iso, 
												 const Vector3D& vcStartGrid, int32 nLayerCubesX, int32 nLayerCubesY, int32 nLayerCubesZ)
{

	// On met a jour les tailles des tableaux intermediaires
	UpdateBufferSizes( nLayerCubesX, nLayerCubesY);

	BoolArray& abCubesInteressantsCurLayer = m_abTesselatedCubePrvLayer;
	BoolArray& abCubesInteressantsNxtLayer = m_abTesselatedCubeCurLayer;

	AllScansArray aScansClean = m_aScansPotentialCubesNxt;


	const Vector3D& vcIncrement = GetCubeElementSize();

	// Demande a l'isosurface une premiere approximation (limites) des cubes qu'il faudra visiter pour creer les vertices
	// ==> resultats dans m_aScansPotentialCubesCur
	iso.FindInterestingPartsOfPlane( m_aScansPotentialCubesCur, vcStartGrid, vcIncrement, nLayerCubesX, nLayerCubesY);

	// Tous les cubes sont declares sans triangles
//	ClearInterestCubes(pabCubesInteressantsCurLayer, m_aScansPotentialCubesCur, nLayerCubesX);

#ifdef _DEBUG
	std::fill( m_afPotentialsPrvLayer.begin(), m_afPotentialsPrvLayer.end(), std::numeric_limits<DensityArray::value_type>::min());

	// Les tableaux de booleens doivent etre vides
	for( int32 j = 0; j < nLayerCubesY; ++j)
		for( int32 i = 0; i < nLayerCubesX; ++i)
		{
			assert( abCubesInteressantsCurLayer[j*nLayerCubesX + i] == false);
			assert( abCubesInteressantsNxtLayer[j*nLayerCubesX + i] == false);
		}
#endif

	// On charge d'abord les potentiels (energie - seuil) sur le plan Z extreme (Z minimum) du grand cube des marching cubes
	// Il sera utilise comme "precedent" pour la premiere couche
	assert( !m_afPotentialsPrvLayer.empty() );
	iso.FuncValuesForPlaneGrid(&m_afPotentialsPrvLayer[0], vcStartGrid,
	                           vcIncrement.x, vcIncrement.y, 
	                           nLayerCubesX+1, nLayerCubesY+1, 
	                           &m_aScansPotentialCubesCur);
	
	// Calcul de la toute premiere couche des vertices
	int32 nVerticesGeneratedLastPlane =	ComputeZPlaneVertices( aVertices, m_aiVtxIndexXPrvLayer, m_aiVtxIndexYPrvLayer,
	                                                           abCubesInteressantsCurLayer, m_abDummyLayer,
	                                                           vcStartGrid, nLayerCubesX, nLayerCubesY,
	                                                           m_afPotentialsPrvLayer,
	                                                           m_aScansPotentialCubesCur, iso);
	// TODO : A virer?
	//ClearInterestCubes( pabCubesInteressantsNxtLayer, *pScansCur, nLayerCubesX);

	for( int32 nCurZLayer = 0; nCurZLayer < nLayerCubesZ; ++nCurZLayer)
	{
#ifdef _DEBUG
		// Valeurs par defauts abherrantes
		std::fill( m_aiVtxIndexXCurLayer.begin(), m_aiVtxIndexXCurLayer.end(), -1);
		std::fill( m_aiVtxIndexYCurLayer.begin(), m_aiVtxIndexYCurLayer.end(), -1);
		std::fill( m_aiVtxIndexZLayer.begin(),    m_aiVtxIndexZLayer.end(),    -1);
#endif

		// On calcule les potentiels sur le plan Z courant

		// Demande a l'isosurface une premiere approximation (limites) des cubes qu'il faudra visiter pour creer les vertices
		// ==> resultats dans m_aScansPotentialCubes
		// Note : Il s'agit d'une optimisation. On parcourt juste les cubes potentiellement interessants plutot que tous
		//        La notion de "potentiellement interessant" est laissee a l'appreciation de l'isosurface
		Vector3D vcCornerLowerLayerPlane = vcStartGrid;
		vcCornerLowerLayerPlane.z += float(nCurZLayer) * vcIncrement.z;

		Vector3D vcCornerUpperLayerPlane = vcCornerLowerLayerPlane;
		vcCornerUpperLayerPlane.z += vcIncrement.z;
		
		// Cherche les scanlines de cubes en dessous du plan, en contact avec des boites englobantes de blobs
		iso.FindInterestingPartsOfPlane( m_aScansPotentialCubesNxt, vcCornerUpperLayerPlane, vcIncrement, nLayerCubesX, nLayerCubesY);

		// Optimisation : si la couche est vide, on passe directement a la suivante
// TODO : A remettre
//		if( bEmptyPlane )
//			continue;

#ifdef _DEBUG
		std::fill( m_afPotentialsCurLayer.begin(), m_afPotentialsCurLayer.end(), MIN_FLOAT32);
#endif

		// Par defaut : calcule tous les potentiels sans passer par les scanlines
		const AllScansArray* pLayerScansToGetValues = 0;
	
		if( nLayerCubesX * nLayerCubesY > 64*64)
		{
				// Optimisation : ne calculer les valeur que la ou elles seront utilisees 
			iso.FindInterestingPartsOfPlane( aScansClean, vcCornerLowerLayerPlane, vcIncrement + Vector3D(0,0,vcIncrement.z), nLayerCubesX, nLayerCubesY);
//			aScansClean = aScansCur;
//			aScansClean.ReInit();
//			aScansClean.Merge(aScansCur);
//			aScansClean.Merge(aScansNxt);
			pLayerScansToGetValues = &aScansClean;
		}
		assert( !m_afPotentialsCurLayer.empty() );
		iso.FuncValuesForPlaneGrid(&m_afPotentialsCurLayer[0], vcCornerUpperLayerPlane,
			                        vcIncrement.x, vcIncrement.y, 
									nLayerCubesX+1, nLayerCubesY+1, 
									pLayerScansToGetValues);

//		LoadZPlanePotentials( pafPotentialsCurLayer, iso, vcCornerUpperLayerPlane, pLayerScansToGetValues );


		// Calcul des nouveaux vertex sur les aretes en axe Z a travers la couche courante
		int32 nVerticesGeneratedNewLayer = ComputeZVerticesThroughLayer( aVertices, m_aiVtxIndexZLayer,
		                                                                 abCubesInteressantsCurLayer,
																		 vcCornerLowerLayerPlane, nLayerCubesX, nLayerCubesY,
																		 m_afPotentialsPrvLayer, m_afPotentialsCurLayer,
																		 m_aScansPotentialCubesCur, iso);

		// Optimisation : pas tenter de creer des triangles si aucune arete n'est creee a la verticale ni au plan precedent
		// Cette optimisation evite simplement de rentrer dans la boucle de calcul des nouveaux triangles
		if( nVerticesGeneratedNewLayer != 0 || nVerticesGeneratedLastPlane != 0 )
		{
			// On connait a present tous les potentiels necessaires pour la couche courante	
			// Calcul des nouveaux vertex sur les aretes en axes X et Y pour le nouveau plan de Z constant 
			nVerticesGeneratedLastPlane = ComputeZPlaneVertices( aVertices, m_aiVtxIndexXCurLayer, m_aiVtxIndexYCurLayer,
																 abCubesInteressantsNxtLayer, abCubesInteressantsCurLayer,
																 vcCornerUpperLayerPlane, nLayerCubesX, nLayerCubesY,
																 m_afPotentialsCurLayer,
                                                                 m_aScansPotentialCubesCur, iso);
			
			// Cree les triangles pour chaque cube de la couche de cubes
			CreateTrianglesForLayer( aTriangles, abCubesInteressantsCurLayer, nLayerCubesX, // nLayerCubesY,
									 aVertices, m_aScansPotentialCubesCur);
		}
		else
		{
			ClearInterestCubes( abCubesInteressantsCurLayer, m_aScansPotentialCubesCur, nLayerCubesX);
		}

#ifdef _DEBUG
		// Les tableaux de booleens doivent etre vides
		for( int32 j = 0; j < nLayerCubesY; ++j)
			for( int32 i = 0; i < nLayerCubesX; ++i)
			{
                //assert( abCubesInteressantsCurLayer[j*nLayerCubesX + i] == false );
                assert( abCubesInteressantsCurLayer[j*nLayerCubesX + i] == '\0' );
			}
#endif

		// On echange maintenant les buffers
		m_aiVtxIndexXPrvLayer.swap( m_aiVtxIndexXCurLayer );
		m_aiVtxIndexYPrvLayer.swap( m_aiVtxIndexYCurLayer );
		m_afPotentialsPrvLayer.swap( m_afPotentialsCurLayer );

		abCubesInteressantsNxtLayer.swap( abCubesInteressantsCurLayer );

		m_aScansPotentialCubesCur.Swap( m_aScansPotentialCubesNxt );
	
	}

	// Les 2 tableaux de "Cubes interessants" doivent etre initialises a faux pour la fois suivante
	ClearInterestCubes( abCubesInteressantsCurLayer, m_aScansPotentialCubesCur, nLayerCubesX);
}





// Genere les vertices d'un plan de potentiels fourni en parametre, et retourne leur nombre
int32 MarchCubesByLayers::ComputeZPlaneVertices( Vect3DArray& aVertices,
                                                 VtxIndexArray& aiVtxPlaneXAxis, VtxIndexArray& aiVtxPlaneYAxis,
                                                 BoolArray& abCubesInteressantsCurLayer, BoolArray& abCubesInteressantsPrvLayer,
                                                 const Vector3D& vcLayerCorner, int32 nCubesX, int32 nCubesY,
												 const DensityArray& afPlanePotentials, 
												 const AllScansArray& aScansToVisit,
												 const IsoSurface& iso) const
{
	int32 nCurGeneratedVertices = aVertices.size();

	const Vector3D& vcElemCubeSize = GetCubeElementSize();
	const LengthType rSizeCubeX = vcElemCubeSize.x;
	const LengthType rSizeCubeY = vcElemCubeSize.y;

	const int32 nCubesInLayerX = nCubesX;
	const int32 nPointsInLayerX = (nCubesInLayerX + 1);

	const int32 nCubesInLayerY = nCubesY;
	const int32 nPointsInLayerY = (nCubesInLayerY + 1);

	const LengthType rCurZ = vcLayerCorner.z;

	const OneLineScansArray& slLinesScans = aScansToVisit.GetLinesScans();
	int32 nLinesRangesCount = slLinesScans.GetNbScans();

	for( int32 idxLineScan = 0; idxLineScan < nLinesRangesCount; ++idxLineScan)
	{
		const ScanLineRange& curLinesRange = slLinesScans[idxLineScan];
		int32 nBegY = curLinesRange.nRangeBeg;
		int32 nEndY = curLinesRange.nRangeEnd + 1;
		assert(  (idxLineScan == nLinesRangesCount-1) || nEndY != slLinesScans[idxLineScan].nRangeBeg);

		// Si la premiere rangee de cube est potentiellement interessante, il faut commencer a son vertex inferieur
		// sinon le premier vertex est le vertex superieur au cube
		if( nBegY > 0)
			++nBegY;

		for( int32 j = nBegY; j <= nEndY; ++j)
		{
			// 1re et 2e rangees de points -> premier scanline
			int32 nCurLine = std::max(0, j-1);
			const OneLineScansArray& curLine = aScansToVisit[nCurLine];
			int32 nCurLineScansCount = curLine.GetNbScans();
			assert(  nCurLineScansCount > 0);

			int32 nStartPointIndexLine = j * nPointsInLayerX;

			// On Parcourt les scans de cubes a visiter
			for( int32 iScan = 0; iScan < nCurLineScansCount; ++iScan)
			{
				const ScanLineRange& curScan = curLine[iScan];
				int32 nBegScanPoint = curScan.nRangeBeg;
				int32 nEndScanPoint = curScan.nRangeEnd + 1;
				// Si le scanline est colle a gauche, des vertices seront peut etre a creer sur le plan de gauche
				// Sinon on peut commencer a droite du premier cube du scanline
				if( nBegScanPoint > 0)
					++nBegScanPoint;

				int32 nCurPotentialPointIndex = nStartPointIndexLine + nBegScanPoint;
				// Pour chaque scan on visite tous les cubes
				for( int32 i = nBegScanPoint; i <= nEndScanPoint; ++i, ++nCurPotentialPointIndex)
				{
					const float& fPotentialNextXY = afPlanePotentials[nCurPotentialPointIndex];

					assert( i>=0 && i < nPointsInLayerX && j>=0 && j < nPointsInLayerY);

					if( i != 0 )
					{
						const float& fPotentialCurX = afPlanePotentials[nCurPotentialPointIndex - 1];
						assert( fPotentialCurX > -1E15f);

						// Si les deux energies ne sont pas du meme cote du seuil (un plus grand l'autre plus petit),
						// alors il faut creer un vertex
						if( AreFloatsDiffSigns( fPotentialCurX, fPotentialNextXY) )
						{
							assert( fPotentialCurX > -1E15f && fPotentialNextXY > -1E15f);
							int32 iDecal = i - 1;
							// Vertex a creer <==> les cubes alentours genereront des triangles
							int32 idxCube = (j-1) * nCubesInLayerX + iDecal;
							if( j != 0)
							{
								abCubesInteressantsCurLayer[idxCube] = true;
								abCubesInteressantsPrvLayer[idxCube] = true;
							}
							if( j != nPointsInLayerY-1)
							{
								abCubesInteressantsCurLayer[idxCube + nCubesInLayerX] = true;
								abCubesInteressantsPrvLayer[idxCube + nCubesInLayerX] = true;
							}

							// Calcul des coordonnees du nouveau vertex
							LengthType rCurX  = vcLayerCorner.x + (float) iDecal * rSizeCubeX;
							LengthType rCurY  = vcLayerCorner.y + (float) j * rSizeCubeY;

							float fValAtMin = fPotentialCurX;
							float fValAtMax = fPotentialNextXY;

							Vector3D vtxMin( rCurX,              rCurY, rCurZ );
							Vector3D vtxMax( rCurX + rSizeCubeX, rCurY, rCurZ );

							Vector3D vtxNew = vtxMin;
							vtxNew.x += rSizeCubeX * (fValAtMin) / (fValAtMin - fValAtMax );

							for( int32 nLev = 0; nLev < m_nRefineIterCount; ++nLev)
							{
								float fNewVal = iso.FuncValueAtPos( vtxNew );
								
								// Meme signe que minX, minX devient la val intermediaire
								if( fNewVal * fValAtMin >= 0.f )
								{
									vtxMin = vtxNew;
									fValAtMin = fNewVal;
								}
								else
								{
									vtxMax = vtxNew;
									fValAtMax = fNewVal;
								}
								vtxNew.x = vtxMin.x + (vtxMax.x - vtxMin.x) * (fValAtMin) / (fValAtMin - fValAtMax );
							}
							
							// Ajout du nouveau vertex dans le tableau des vertices de l'objet
							int32 iNewVtx = aVertices.size();
							aVertices.push_back( vtxNew );

							// Et on le stocke dans le tableau des vertices crees dans le plan de Z constant,
							// pour assurer son unicite lors de la creation des triangles
							int32 iVtxPlaneIndex = j * nCubesInLayerX + iDecal;
							aiVtxPlaneXAxis[iVtxPlaneIndex] = iNewVtx;
						}
					}

					if( j != 0 )
					{
						const float& fPotentialCurY = afPlanePotentials[nCurPotentialPointIndex - nPointsInLayerX ];
						assert( fPotentialCurY > -1E15f);

						// Si les deux energies ne sont pas du meme cote du seuil (un plus grand l'autre plus petit),
						// alors il faut creer un vertex
						if( AreFloatsDiffSigns( fPotentialCurY, fPotentialNextXY) )
						{
							assert( fPotentialCurY > -1E15f && fPotentialNextXY > -1E15f);
							int32 jDecal = j - 1;
							// Vertex a creer <==> les cubes alentours genereront des triangles
							int32 idxCube = jDecal * nCubesInLayerX + (i - 1);
							if( i != 0)
							{
								abCubesInteressantsCurLayer[idxCube] = true;
								abCubesInteressantsPrvLayer[idxCube] = true;
							}
							if(i != nPointsInLayerX-1)
							{
								abCubesInteressantsCurLayer[idxCube + 1] = true;
								abCubesInteressantsPrvLayer[idxCube + 1] = true;
							}

							// Calcul des coordonnees du nouveau vertex
							LengthType rCurX  = vcLayerCorner.x + (float) i * rSizeCubeX;
							LengthType rCurY  = vcLayerCorner.y + (float) jDecal * rSizeCubeY;

							float fValAtMin = fPotentialCurY;
							float fValAtMax = fPotentialNextXY;

							Vector3D vtxMin( rCurX,              rCurY, rCurZ );
							Vector3D vtxMax( rCurX, rCurY + rSizeCubeY, rCurZ );

							Vector3D vtxNew = vtxMin;
							vtxNew.y += rSizeCubeY * (fValAtMin) / (fValAtMin - fValAtMax );

							for( int32 nLev = 0; nLev < m_nRefineIterCount; ++nLev)
							{
								float fNewVal = iso.FuncValueAtPos( vtxNew );
								
								// Meme signe que minX, minX devient la val intermediaire
								if( fNewVal * fValAtMin >= 0.f )
								{
									vtxMin = vtxNew;
									fValAtMin = fNewVal;
								}
								else
								{
									vtxMax = vtxNew;
									fValAtMax = fNewVal;
								}
								vtxNew.y = vtxMin.y + (vtxMax.y - vtxMin.y) * (fValAtMin) / (fValAtMin - fValAtMax );
							}

							// Ajout du nouveau vertex dans le tableau des vertices de l'objet
							int32 iNewVtx = aVertices.size();
							aVertices.push_back( vtxNew );

							// Et on le stocke dans le tableau des vertices crees dans le plan de Z constant,
							// pour assurer son unicite lors de la creation des triangles
							int32 iVtxPlaneIndex = jDecal * nPointsInLayerX + i;
							aiVtxPlaneYAxis[iVtxPlaneIndex] = iNewVtx;
						}
					}
				}
			}
		}
	}
    //  Creation des vertices sur les aretes d'axe Y, sur le plan Z constant donne
	return aVertices.size() - nCurGeneratedVertices;
}








// Calcul des nouveaux vertex sur les aretes en axe Z a travers la couche courante et retourne le nb de nvx vertices
int32 MarchCubesByLayers::ComputeZVerticesThroughLayer( Vect3DArray& aVertices, VtxIndexArray& aiVtxIndexZLayer,
                                                       BoolArray&  abCubesInteressantsCurLayer,
													   const Vector3D& vcLowLayerCorner, int32 nCubesX, int32 nCubesY,
													   const DensityArray& afPlane1Potentials, const DensityArray& afPlane2Potentials, 
                                                       const AllScansArray& aScansToVisit,
                                                       const IsoSurface& iso) const

{
	int32 nCurGeneratedVertices = aVertices.size();

	const Vector3D& vcElemCubeSize = GetCubeElementSize();
	const LengthType rSizeCubeX = vcElemCubeSize.x;
	const LengthType rSizeCubeY = vcElemCubeSize.y;
	const LengthType rSizeCubeZ = vcElemCubeSize.z;

	const int32 nPointsInLayerX = nCubesX + 1;
	const int32 nPointsInLayerY = nCubesY + 1;
	const int32 nCubesInLayerX = nCubesX;

	const OneLineScansArray& slLinesScans = aScansToVisit.GetLinesScans();
	int32 nLinesRangesCount = slLinesScans.GetNbScans();

	for( int32 idxLineScan = 0; idxLineScan < nLinesRangesCount; ++idxLineScan)
	{
		const ScanLineRange& curLinesRange = slLinesScans[idxLineScan];
		int32 nBegY = curLinesRange.nRangeBeg;
		int32 nEndY = curLinesRange.nRangeEnd + 1;
		assert(  (idxLineScan == nLinesRangesCount-1) || nEndY != slLinesScans[idxLineScan].nRangeBeg);

		for( int32 j = nBegY; j <= nEndY; ++j)
		{
			int32 nCurLine = std::min(j, nEndY-1);
			const OneLineScansArray& curLine = aScansToVisit[nCurLine];
			int32 nCurLineScansCount = curLine.GetNbScans();
			assert(  nCurLineScansCount > 0);

			int32 nStartPointIndexLine = j * nPointsInLayerX;

			// On Parcourt les scans de cubes a visiter
			for( int32 iScan = 0; iScan < nCurLineScansCount; ++iScan)
			{
				const ScanLineRange& curScan = curLine[iScan];
				int32 nBegScanPoint, nEndScanPoint;

				nBegScanPoint = curScan.nRangeBeg;
				nEndScanPoint = curScan.nRangeEnd+1;
				assert( nEndScanPoint < nPointsInLayerX);
				int32 nCurPotentialPointIndex = nStartPointIndexLine + nBegScanPoint;
				// Pour chaque scan on visite tous les cubes

				for( int32 i = nBegScanPoint; i <= nEndScanPoint; ++i, ++nCurPotentialPointIndex)
				{
					const float& fPotentialZ1 = afPlane1Potentials[nCurPotentialPointIndex];
					const float& fPotentialZ2 = afPlane2Potentials[nCurPotentialPointIndex];

					assert( fPotentialZ1 > -1E15f && fPotentialZ2 > -1E15f);

					// Si les deux energies ne sont pas du meme cote du seuil (un plus grand l'autre plus petit),
					// alors il faut creer un vertex
					if( AreFloatsDiffSigns( fPotentialZ1, fPotentialZ2) )
					{
						assert( i>=0 && i < nPointsInLayerX && j>=0 && j < nPointsInLayerY);
						// Vertex a creer <==> les cubes alentours genereront des triangles
						int32 idxCube = (j-1) * nCubesInLayerX + (i-1);
						if( j != 0)
						{
							if( i != 0)
								abCubesInteressantsCurLayer[idxCube] = true;
							if( i != nPointsInLayerX-1)
								abCubesInteressantsCurLayer[idxCube + 1] = true;
						}
						if( j != nPointsInLayerY-1)
						{
							if( i != 0)
								abCubesInteressantsCurLayer[idxCube + nCubesInLayerX] = true;
							if( i != nPointsInLayerX-1)
								abCubesInteressantsCurLayer[idxCube + nCubesInLayerX + 1] = true;
						}

						// Calcul des coordonnees du nouveau vertex
						LengthType rCurX  = vcLowLayerCorner.x + float(i) * rSizeCubeX;
						LengthType rCurY  = vcLowLayerCorner.y + float(j) * rSizeCubeY;
						LengthType rCurZ  = vcLowLayerCorner.z;


						float fValAtMin = fPotentialZ1;
						float fValAtMax = fPotentialZ2;

						Vector3D vtxMin( rCurX, rCurY, rCurZ );
						Vector3D vtxMax( rCurX, rCurY, rCurZ + rSizeCubeZ);

						Vector3D vtxNew = vtxMin;
						vtxNew.z += rSizeCubeZ * (fValAtMin) / (fValAtMin - fValAtMax );

						for( int32 nLev = 0; nLev < m_nRefineIterCount; ++nLev)
						{
							float fNewVal = iso.FuncValueAtPos( vtxNew );

							// Meme signe que minX, minX devient la val intermediaire
							if( fNewVal * fValAtMin >= 0.f )
							{
								vtxMin = vtxNew;
								fValAtMin = fNewVal;
							}
							else
							{
								vtxMax = vtxNew;
								fValAtMax = fNewVal;
							}
							vtxNew.z = vtxMin.z + (vtxMax.z - vtxMin.z) * (fValAtMin) / (fValAtMin - fValAtMax );
						}

						// Ajout du nouveau vertex dans le tableau des vertices de l'objet
						int32 iNewVtx = aVertices.size();
						aVertices.push_back( vtxNew  );

						// Et on le stocke dans le tableau des vertices crees sur les aretes Z de la couche courante
						int32 iVtxPlaneIndex = j * nPointsInLayerX + i;
						aiVtxIndexZLayer[iVtxPlaneIndex] = iNewVtx;
					}
				}
			}
		}
	}
	return aVertices.size() - nCurGeneratedVertices;
}


void MarchCubesByLayers::CreateTrianglesForLayer( TriangleArray& aTriangles,
													BoolArray& abInterestCubesCurLayer, int32 nCubesX, // int32 nCubesY,
													const Vect3DArray&  aVertices ,            // utilise en mode debug
													const AllScansArray& aScansToVisit) const
{

	const DensityArray& afPotentialsZInf = m_afPotentialsPrvLayer;
	const DensityArray& afPotentialsZSup = m_afPotentialsCurLayer;

	const VtxIndexArray& aiVtxIndexXInf   = m_aiVtxIndexXPrvLayer;
	const VtxIndexArray& aiVtxIndexXSup   = m_aiVtxIndexXCurLayer;
	const VtxIndexArray& aiVtxIndexYInf   = m_aiVtxIndexYPrvLayer;
	const VtxIndexArray& aiVtxIndexYSup   = m_aiVtxIndexYCurLayer;
	const VtxIndexArray& aiVtxIndexZLayer = m_aiVtxIndexZLayer;


	const int32 nCubesInLayerX = nCubesX;
	const int32 nPointsInLayerX = nCubesInLayerX + 1;

	const OneLineScansArray& slLinesScans = aScansToVisit.GetLinesScans();
	int32 nLinesRangesCount = slLinesScans.GetNbScans();
	
	for( int32 idxLineScan = 0; idxLineScan < nLinesRangesCount; ++idxLineScan)
	{
		const ScanLineRange& curLinesRange = slLinesScans[idxLineScan];
		int32 nBegY = curLinesRange.nRangeBeg;
		int32 nEndY = curLinesRange.nRangeEnd;

		for( int32 iCubeY = nBegY; iCubeY <= nEndY; ++iCubeY)
		{
			// curLine : ligne courante des scanlines a visiter
			const OneLineScansArray& curLine = aScansToVisit[iCubeY];
			int32 nStartLineIndex = iCubeY * nCubesInLayerX;

			int32 iSavFirstCubePointX1 = iCubeY*nPointsInLayerX;

			// On se balade dans les scans pour ne modifier les gros tableaux 2D que la ou c'est necessaire
			int32 nCurLineScansCount = curLine.GetNbScans();
			for( int32 iScan = 0; iScan < nCurLineScansCount; ++iScan)
			{
				const ScanLineRange& curScan = curLine[iScan];
				int32 nIndexCubeDeb = nStartLineIndex + curScan.nRangeBeg;
				int32 nIndexCubeFin = nStartLineIndex + curScan.nRangeEnd;

				// Boucle interne a un scan
				for(int32 nIndexCube = nIndexCubeDeb ; nIndexCube <= nIndexCubeFin; ++nIndexCube)
				{
					// Des triangles ne sont crees que si tous les sommets du marching cube n'ont pas une valeur de meme signe
					// On ne peut rentrer dans ce bloc que si le cube est interessant (generera des triangles)
					BoolArray::reference bCurCubeInteressant = abInterestCubesCurLayer[nIndexCube];
					if( bCurCubeInteressant )
					{
						int32 iCubeX = nIndexCube - nIndexCubeDeb + curScan.nRangeBeg;
						int32 iFirstCubePointX1 = iSavFirstCubePointX1 + iCubeX;

						bCurCubeInteressant = false;
						int32 iFirstCubePointX2 = iFirstCubePointX1 + nPointsInLayerX;
						const float& fDensityCubePoint0 = afPotentialsZInf[iFirstCubePointX1    ];
						const float& fDensityCubePoint1 = afPotentialsZInf[iFirstCubePointX1 + 1];
						const float& fDensityCubePoint3 = afPotentialsZSup[iFirstCubePointX1    ];
						const float& fDensityCubePoint2 = afPotentialsZSup[iFirstCubePointX1 + 1];

						const float& fDensityCubePoint4 = afPotentialsZInf[iFirstCubePointX2    ];
						const float& fDensityCubePoint5 = afPotentialsZInf[iFirstCubePointX2 + 1];
						const float& fDensityCubePoint7 = afPotentialsZSup[iFirstCubePointX2    ];
						const float& fDensityCubePoint6 = afPotentialsZSup[iFirstCubePointX2 + 1];

						int32 nCubeConfig = FastFloatGetSign(fDensityCubePoint0)
										 | (FastFloatGetSign(fDensityCubePoint1) << 1)
										 | (FastFloatGetSign(fDensityCubePoint2) << 2)
										 | (FastFloatGetSign(fDensityCubePoint3) << 3)
										 | (FastFloatGetSign(fDensityCubePoint4) << 4)
										 | (FastFloatGetSign(fDensityCubePoint5) << 5)
										 | (FastFloatGetSign(fDensityCubePoint6) << 6)
										 | (FastFloatGetSign(fDensityCubePoint7) << 7);

						// On ne pouvait rentrer dans ce bloc que si le cube est interessant
						//			Signes differents ==> nCubeConfig different de 0 et de 0xFF, ce qui s'exprime :
						assert( ((nCubeConfig +1) & 0xFE) != 0);

						const int32 nVtxX1IndexOffset = iCubeY * nCubesInLayerX + iCubeX;
						const int32 nVtxX2IndexOffset = nVtxX1IndexOffset + nCubesInLayerX;

						const int32 nVtxY1IndexOffset = iCubeY * nPointsInLayerX + iCubeX;
						const int32 nVtxY2IndexOffset = nVtxY1IndexOffset + 1;

						const int32 nVtxZ1IndexOffset = nVtxY1IndexOffset;
						const int32 nVtxZ2IndexOffset = nVtxZ1IndexOffset + 1;
						const int32 nVtxZ3IndexOffset = nVtxZ2IndexOffset + nPointsInLayerX;
						const int32 nVtxZ4IndexOffset = nVtxZ1IndexOffset + nPointsInLayerX;

						// Tableau des indices des vertices crees, pour le marching cube courant
						// On prend les adresses pour eviter les indirections inutiles
						const int32* const apiVtxToUse[12] = 
						{
							// Aretes du cube (Y Inf)
							&(aiVtxIndexXInf  [ nVtxX1IndexOffset]),
							&(aiVtxIndexZLayer[ nVtxZ2IndexOffset]),
							&(aiVtxIndexXSup  [ nVtxX1IndexOffset]),
							&(aiVtxIndexZLayer[ nVtxZ1IndexOffset]),

							// Aretes du cube (Y Sup)
							&(aiVtxIndexXInf  [ nVtxX2IndexOffset]),
							&(aiVtxIndexZLayer[ nVtxZ3IndexOffset]),
							&(aiVtxIndexXSup  [ nVtxX2IndexOffset]),
							&(aiVtxIndexZLayer[ nVtxZ4IndexOffset]),

							// Aretes du cube (a travers Y)
							&(aiVtxIndexYInf  [ nVtxY1IndexOffset]),
							&(aiVtxIndexYInf  [ nVtxY2IndexOffset]),
							&(aiVtxIndexYSup  [ nVtxY2IndexOffset]),
							&(aiVtxIndexYSup  [ nVtxY1IndexOffset])
						};

						const SCubeConf& triangleConf = k_aCubeConfs[nCubeConfig];
						int32 nTris = triangleConf.nTriCount;

//						int32 nIdxFirstNewTri = aTriangles.size();
//						aTriangles.resize( nIdxFirstNewTri + nTris );

						for( int32 iTri = 0; iTri < nTris; ++iTri)
						{
							// curTri est le triangle courant a creer, dont les indices de vertex sont ceux
							// des aretes correspondantes dans le cube (entre 0 et 11)
							const SCubeConf::SCubeTriangle& curTri = triangleConf.aTriangles[iTri];

							// On ne doit pas avoir depasse les triangles valides
							assert( curTri.a != -1);

							uint32 nlVtxA = *(apiVtxToUse[curTri.a]);
							uint32 nlVtxB = *(apiVtxToUse[curTri.b]);
							uint32 nlVtxC = *(apiVtxToUse[curTri.c]);

							assert( nlVtxA >= 0 && nlVtxA < aVertices.size());
							assert( nlVtxB >= 0 && nlVtxB < aVertices.size());
							assert( nlVtxC >= 0 && nlVtxC < aVertices.size());

//							Triangle& newTri = aTriangles[nIdxFirstNewTri + iTri];
//							newTri.Set( nlVtxA, nlVtxB, nlVtxC );
							aTriangles.push_back( Triangle( nlVtxA, nlVtxB, nlVtxC ) );
						}
					} // if( bCurCubeInteressant )
				} // Boucle des scans de la ligne
			} // Boucle X sur les cubes
		} // Boucle Y sur les cubes
	} // Boucle sur les scans (idxLineScan)
}



void MarchCubesByLayers::ClearInterestCubes( BoolArray& abCubes, const AllScansArray& aScansToVisit, int32 nCubesX) 
{

	const OneLineScansArray& slLinesScans = aScansToVisit.GetLinesScans();
	int32 nLinesRangesCount = slLinesScans.GetNbScans();

	for( int32 idxLineScan = 0; idxLineScan < nLinesRangesCount; ++idxLineScan)
	{
		const ScanLineRange& curLinesRange = slLinesScans[idxLineScan];
		int32 nBegY = curLinesRange.nRangeBeg;
		int32 nEndY = curLinesRange.nRangeEnd;

		for( int32 j = nBegY; j <= nEndY; ++j)
		{
			// curLine : ligne courante des scanlines a visiter
			const OneLineScansArray& curLine = aScansToVisit[j];
			int32 nStartLineIndex = j * nCubesX;

			// On se balade dans les scans pour ne modifier les gros tableaux 2D que la ou c'est necessaire
			int32 nCurLineScansCount = curLine.GetNbScans();
			for( int32 i = 0; i < nCurLineScansCount; ++i)
			{
				const ScanLineRange& curScan = curLine[i];
				for( int32 k = curScan.nRangeBeg; k <= curScan.nRangeEnd; ++k)
					abCubes[nStartLineIndex + k] = false;
			}
		}
	}
}










// Calcul des normales
void MarchCubesByLayers::ComputeNormalsFromTopology(Vect3fArray& aNorms, 
													const Vect3DArray& aVerts,
													const TriangleArray& aTris) 
{
	Vect3DArray::size_type    nVertexCount = aVerts.size();
	if(	aNorms.size() != nVertexCount)
		aNorms.resize(nVertexCount);

	// On annule les normales
	std::fill( aNorms.begin(), aNorms.end(), Vector3f(0,0,0) );

	for( TriangleArray::const_iterator itTri = aTris.begin(); itTri != aTris.end(); ++itTri )
	{
		int32 nVtxA = itTri->a;
		int32 nVtxB = itTri->b;
		int32 nVtxC = itTri->c;
		const Vector3D& vcA = aVerts[nVtxA];
		Vector3D vcAB = aVerts[nVtxB]; vcAB -= vcA;
		Vector3D vcAC = aVerts[nVtxC]; vcAC -= vcA;
		
		Vector3f vcNormTri = Vector3f::ConvertFrom( vcAB ^ vcAC );

		aNorms[nVtxA] += vcNormTri;
		aNorms[nVtxB] += vcNormTri;
		aNorms[nVtxC] += vcNormTri;
	}
}



} // namespace



